package com.mx.dla.dda.contrato.transaccion.terminos.bos;

import java.io.IOException;
import java.sql.SQLException;

import org.apache.ibatis.exceptions.PersistenceException;
import org.apache.ibatis.executor.result.ResultMapException;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.mx.dla.dda.contrato.transaccion.exceptions.dtos.TransaccionException;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.ExcepcionesSeccionDTO;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.LibreriaDTO;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.MinimoGarantizadoDTO;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.PrecioRentaDTO;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.RevenueBoxGloDTO;
import com.mx.dla.dda.contrato.transaccion.terminos.dtos.SeccionesTerminos;
import com.mx.dla.global.bos.BaseBO;

@Service
public class TerminosLibreria extends BaseBO {

	@Autowired
	private TerminosMinimos minimos;

	@Autowired
	@Qualifier("precioLibreria")
	private TerminosPrecio precio;

	@Autowired
	private TerminosRevenue revenue;

	@Autowired
	private TerminosExcepciones excepciones;

	public LibreriaDTO obtenerLibreria(Long idContrato) throws ResultMapException, SQLException, TransaccionException {
		LibreriaDTO revenueDTO = new LibreriaDTO();

		try {
			MinimoGarantizadoDTO minimo = minimos.obtenerMinimos(idContrato, SeccionesTerminos.RentaLibreria.getDesc());

			PrecioRentaDTO price = precio.obtenerDatosPreciosLibreria(idContrato,
					SeccionesTerminos.RentaLibreria.getDesc());

			ExcepcionesSeccionDTO excepcionesDTO = excepciones.obtenerExcepciones(idContrato,
					SeccionesTerminos.RentaPremium.getDesc());

			RevenueBoxGloDTO revenueDatos = revenue.obtenerRevenueLibreria(idContrato,
					SeccionesTerminos.RentaLibreria.getDesc());
			revenueDTO.setMinimos(minimo);
			revenueDTO.setPrecio(price);
			revenueDTO.setExcepciones(excepcionesDTO);
			revenueDTO.setRevenue(revenueDatos);
		} catch (PersistenceException e) {
			throw new TransaccionException("Error al cargar los datos.", e);
		} catch (DataAccessException e) {
			throw new TransaccionException("Error al cargar los datos", e);
		}

		return revenueDTO;
	}

	@Transactional(propagation = Propagation.REQUIRED, rollbackFor = TransaccionException.class)
	public void guardarLibreria(LibreriaDTO p, Long idContrato) throws JsonParseException, JsonMappingException,
			IOException, ResultMapException, SQLException, TransaccionException {

		try {
			logger.debug(p.toString());

			revenue.eliminarMinimosCategoriasBoxOffice(idContrato, SeccionesTerminos.RentaLibreria.getDesc());
			revenue.elimnarRevenueGlobal(idContrato, SeccionesTerminos.RentaLibreria.getDesc());
			precio.eliminarPaisesImpuestoEstatus(idContrato, SeccionesTerminos.RentaLibreria.getDesc());
			precio.eliminarPreciosLibreria(idContrato);
			excepciones.eliminarExcepcion(idContrato, SeccionesTerminos.RentaLibreria.getDesc());

			minimos.guardarMinimos(p.getMinimos().getMinimosCateg(), p.getMinimos().getMinimosAnio(),
					p.getMinimos().getAnios(), p.getMinimos().getTipoMin(), SeccionesTerminos.RentaLibreria.getDesc(),
					idContrato);

			precio.actualizarPrecioAplica(idContrato, SeccionesTerminos.RentaLibreria.getDesc(),
					p.getPrecio().getAplica());
			precio.actualizarPagoPpe(idContrato, SeccionesTerminos.RentaLibreria.getDesc(), p.getPrecio().getPagoPPE());

			if (p.getPrecio().getAplica().intValue() == 1)
				precio.guardarDatosPreciosLibreria(p.getPrecio(), idContrato,
						SeccionesTerminos.RentaLibreria.getDesc());

			revenue.guardarRevenueLibreria(idContrato, SeccionesTerminos.RentaLibreria.getDesc(), p.getRevenue());
			excepciones.guardarExcepciones(idContrato, SeccionesTerminos.RentaLibreria.getDesc(), p.getExcepciones());

		} catch (PersistenceException e) {
			throw new TransaccionException("Error al cargar los datos.", e);
		} catch (DataAccessException e) {
			throw new TransaccionException("Error al cargar los datos", e);
		}
	}

}
