package com.mx.dla.dda.facturacion.bos;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

import com.mx.dla.dda.bussiness.utilerias.BussinessUtilerias;
import com.mx.dla.dda.common.exception.ValidacionException;
import com.mx.dla.dda.facturacion.daos.FacturacionDAO;
import com.mx.dla.dda.facturacion.dtos.FacturaSuscriptoresBD;
import com.mx.dla.dda.facturacion.dtos.FacturaTransaccionesBD;
import com.mx.dla.dda.facturacion.dtos.FacturaDTO;
import com.mx.dla.dda.facturacion.dtos.FacturaVerificadaBD;
import com.mx.dla.dda.facturacion.dtos.FacturaVerificadaDTO;
import com.mx.dla.dda.facturacion.dtos.RespuestaDTO;
import com.mx.dla.dda.facturacion.trans.TransformacionSAPToDDA;
import com.mx.dla.dda.utilerias.BeanUtilerias;

public class FacturacionBO {

	private static Logger  log = Logger.getLogger(FacturacionBO.class);

	@Autowired
	private FacturacionDAO facturacionDAO;

	public RespuestaDTO notificarPago(List<FacturaDTO> facturas) {

		// List<RespuestaDTO> items = new ArrayList<RespuestaDTO>();
		RespuestaDTO respuesta = null;
		boolean valido = true;
		List<FacturaSuscriptoresBD> requests = new ArrayList<FacturaSuscriptoresBD>();
		String doctoOc = null;

		try {

			for (FacturaDTO factura : facturas) {
				log.info("    <<<< Datos del facturas >>>");

				BeanUtilerias.trimStrings(factura);
				doctoOc = factura.getDocto_oc();
				
				FacturaSuscriptoresBD facturaBD = null;
				facturaBD = TransformacionSAPToDDA.getFacturaBD(factura);
				
				log.info(facturaBD);

				log.info("    Inicio de la validacion de la factura:" + doctoOc);
				respuesta = verificar(factura);

				if (!respuesta.getEstatus().equals("OK"))
					throw new ValidacionException(respuesta.getDescripcion());

				Long idContratoOriginal = facturacionDAO.getIdContratoOriginalByOrdenCompra(facturaBD.getOrdenCompra());

				if (idContratoOriginal == null)
					throw new ValidacionException("No se encontro contrato asociado a la orden de compra: " + facturaBD.getOrdenCompra());

				facturaBD.setIdContrato(idContratoOriginal);
				requests.add(facturaBD);

				log.info("    Fin de la validacion de la factura:" + doctoOc + " con resultado: " + respuesta.getEstatus() + " - " + respuesta.getDescripcion());

			}

		}
		catch (Exception e) {
			respuesta.setDescripcion(e.getMessage());
			respuesta.setEstatus("ERROR");
			log.info("    Fin de la validacion de la factura: " + doctoOc + " con resultado: " + respuesta.getEstatus() + " - " + respuesta.getDescripcion());

			valido = false;
		}

		if (valido) {
			try {

				for (FacturaSuscriptoresBD factura : requests) {
					List<FacturaSuscriptoresBD> existen = null;
					LinkedList<FacturaTransaccionesBD> auxiliar = null;

					log.info("   Inicia persistencia de la factura:" + factura.getDocto_oc());

					FacturaTransaccionesBD transaccion = new FacturaTransaccionesBD();
					transaccion.setFacturaI14(factura.getDocto_oc());

					List<FacturaTransaccionesBD> transacciones = facturacionDAO.buscarFacturaTransaccional(transaccion);

					if (transacciones != null && transacciones.size() > 0) {
						auxiliar = new LinkedList<FacturaTransaccionesBD>(transacciones);
						FacturaTransaccionesBD item = auxiliar.getFirst();
						item.setDoctoSapI16(factura.getDocto_em());
						facturacionDAO.actualizarFactura(item);
						log.info("      Persistencia factura transaccional: " + item);
					}
					else {
						existen = facturacionDAO.buscaPagoByOrdenAndDocEm(factura);
						if (existen != null && existen.size() > 0)
							facturacionDAO.actualizarPago(factura);
						else
							facturacionDAO.notificarPago(factura);
						
						log.info("      Persistencia factura suscriptor: " + factura);

					}

					log.info("    Fin persistencia de la factura:" + factura.getDocto_oc() + " con resultado: " + respuesta.getEstatus() + " - " + respuesta.getDescripcion());

				}
			}
			catch (Exception e) {
				respuesta.setEstatus("Error-501");
				respuesta.setDescripcion("Error Servidor Facturacion." + e);
				log.error("Error Servidor Facturacion." + e);
			}

		}
		return respuesta;
	}

	public RespuestaDTO notificarFacturaVerificada(List<FacturaVerificadaDTO> facturas) {

		// List<RespuestaFacturaDTO> response = new
		// ArrayList<RespuestaFacturaDTO>();
		RespuestaDTO respuesta = null;
		boolean valido = true;
		String facturaNo = null;
		List<FacturaVerificadaBD> requests = new ArrayList<FacturaVerificadaBD>();

		try {

			for (FacturaVerificadaDTO factura : facturas) {
				respuesta = null;
				log.info("    <<<< Datos del estudio >>>");
				log.info(factura.toString());

				BeanUtilerias.trimStrings(factura);
				facturaNo = factura.getNumFactura();

				log.info("    Inicio de la validacion de la factura:" + factura.getNumFactura());
				respuesta = this.verificarFactura(factura);

				if (!respuesta.getEstatus().equals("OK"))
					throw new ValidacionException(respuesta.getDescripcion());

				FacturaVerificadaBD facturaBD = null;
				facturaBD = TransformacionSAPToDDA.getFacturaVerificadaBD(factura, respuesta);

				if (!BussinessUtilerias.validaSapNegativo(facturaBD.getImporte())) {
					List<FacturaVerificadaBD> existen = facturacionDAO.buscaContratoMaterial(facturaBD);
					if (existen == null || existen.size() <= 0)
						throw new ValidacionException("No se encontro registro de materiales asociado a la orden de compra: " + factura.getOrdenCompra());
				}

				requests.add(facturaBD);

				log.info(facturaBD);
				log.info("    Fin de la validacion del factura:" + facturaNo + " con resultado: " + respuesta.getEstatus());

			}

		}
		catch (Exception e) {
			respuesta.setDescripcion(e.getMessage());
			respuesta.setEstatus("ERROR");
			log.info("    Fin de la validacion de la factura: " + facturaNo + " con resultado: " + respuesta.getEstatus() + " - " + respuesta.getDescripcion());

			valido = false;
		}

		if (valido) {
			try {
				for (FacturaVerificadaBD factura : requests) {
					facturacionDAO.notificarFacturaVerificada(factura);

					if (!BussinessUtilerias.validaSapNegativo(factura.getImporte()))
						facturacionDAO.actualizaContratoMaterial(factura);
				}
			}
			catch (Exception e) {
				respuesta.setEstatus("Error-501");
				respuesta.setDescripcion("Error inesperado");
				log.error("Error Servidor Facturacion." + e);
			}

		}

		return respuesta;
	}

	public RespuestaDTO verificar(FacturaDTO factura) {

		RespuestaDTO respuesta = new RespuestaDTO();
		respuesta.setEstatus("OK");
		respuesta.setDescripcion("Validacion Exitosa");

		if (factura.getSociedad() == null || factura.getSociedad().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros [sociedad]");
		}
		else if (factura.getSociedad().length() > 4) {
			respuesta.setEstatus("DDA-202");
			// factura.setSociedad(this.verificarLongitud(factura.getSociedad(), 4));
			respuesta.setDescripcion("El parametro [sociedad] no cumple con el formato requerido (max 4 caracteres)");
		}

		if (factura.getAnnio() == null || factura.getAnnio().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros [annio]");
		}
		else if (factura.getAnnio().length() > 4 || !this.verificarAnnio(factura.getAnnio())) {
			respuesta.setEstatus("DDA-202");
			// factura.setAnnio(this.verificarLongitud(factura.getAnnio(), 4));
			respuesta.setDescripcion("El parametro [annio] no cumple con el formato requerido (max 4 caracteres numericos)");
		}

		if (factura.getImporte() == null || factura.getImporte().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros [importe]");
		}
		else if (factura.getImporte().length() > 15 || !BussinessUtilerias.verificarNumero(factura.getImporte().trim())) {
			respuesta.setEstatus("DDA-202");
			// factura.setImporte(this.verificarLongitud(factura.getImporte(), 15));
			respuesta.setDescripcion("El parametro [importe] no cumple con el formato requerido char(13,2)");
		}

		if (factura.getDocto_oc() == null || factura.getDocto_oc().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros [docto_oc]");
		}
		else if (factura.getDocto_oc().length() > 10) {
			respuesta.setEstatus("DDA-202");
			// factura.setDocto_oc(this.verificarLongitud(factura.getDocto_oc(), 10));
			respuesta.setDescripcion("El parametro [docto_oc] no cumple con el formato requerido (max 10 caracteres)");
		}

		if (factura.getDocto_em() == null || factura.getDocto_em().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros [docto_em]");
		}
		else if (factura.getDocto_em().length() > 14) {
			respuesta.setEstatus("DDA-202");
			// factura.setDocto_em(this.verificarLongitud(factura.getDocto_em(), 14));
			respuesta.setDescripcion("El parametro [docto_em] no cumple con el formato requerido (max 14 caracteres)");
		}

		return respuesta;
	}

	public RespuestaDTO verificarFactura(FacturaVerificadaDTO factura) {
		RespuestaDTO respuesta = new RespuestaDTO();
		respuesta.setEstatus("OK");
		respuesta.setDescripcion("Datos registrados correctamente en DDA");

		if (factura.getImporte() == null || factura.getImporte().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros[importe]");
		}
		else if (!BussinessUtilerias.validaImporte(factura.getImporte().trim())) {
			respuesta.setEstatus("DDA-202");
			respuesta.setDescripcion("El par�metro [importe] no empata con el formado 13 o 1 enteros . 2 o 0 decimales");
			factura.setImporte(this.verificarLongitud(factura.getImporte(), 15));
		}

		if (factura.getNumDocto() == null || factura.getNumDocto().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros[numDocto]");
		}
		else if (factura.getNumDocto().length() > 10) {
			respuesta.setEstatus("DDA-202");
			respuesta.setDescripcion("El par�metro [numDocto] excede tama�o");
			factura.setNumDocto(this.verificarLongitud(factura.getNumDocto(), 10));
		}

		if (factura.getNumFactura() == null || factura.getNumFactura().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros[numFactura]");
		}
		else if (factura.getNumFactura().length() > 16) {
			respuesta.setEstatus("DDA-202");
			respuesta.setDescripcion("El par�metro [numFactura] excede tama�o");
			factura.setNumFactura(this.verificarLongitud(factura.getNumFactura(), 16));
		}

		if (factura.getOrdenCompra() == null || factura.getOrdenCompra().equals("")) {
			respuesta.setEstatus("DDA-201");
			respuesta.setDescripcion("Falta informacion en parametros[ordenCompra]");
		}
		else if (factura.getOrdenCompra().length() > 10) {
			respuesta.setEstatus("DDA-202");
			respuesta.setDescripcion("El par�metro [ordenCompra] excede tama�o");
			factura.setNumFactura(this.verificarLongitud(factura.getOrdenCompra(), 10));
		}

		return respuesta;
	}

	public String verificarLongitud(String str, int tamano) {
		if (str.length() > tamano)
			return str.substring(0, tamano);
		else
			return str;
	}

	public boolean verificarAnnio(String fecha) {
		return Pattern.matches("\\d\\d\\d\\d", fecha);
	}


}
